//--------------------------------------------------------------
// Shapes
//
// Copyright (c) LWKS Software Ltd.  All Rights Reserved
//--------------------------------------------------------------
#include "_utils.fx"

DeclareLightworksEffect( "Simple 2D Shape", "Mattes", "Components", kNoNotes, "ScaleAware|HasMinOutputSize" );

//--------------------------------------------------------------
// Params
//--------------------------------------------------------------
DeclareIntParam( SetTechnique, "Shape", kNoGroup, 0, "Rectangle|Ellipse" );
DeclareFloatParam( CentreX,  "Position", kNoGroup, "SpecifiesPointX|DisplayAsPercentage", 0.5, -1.0, 2.0 );
DeclareFloatParam( CentreY,  "Position", kNoGroup, "SpecifiesPointY|DisplayAsPercentage", 0.5, -1.0, 2.0 );
DeclareFloatParam( Width,    "Width",    kNoGroup, "DisplayAsPercentage",                 0.5, 0.00001, 1.00 );
DeclareFloatParam( Height,   "Height",   kNoGroup, "DisplayAsPercentage",                 0.5, 0.00001, 1.00 );
DeclareFloatParam( Softness, "Softness", kNoGroup, "DisplayAsPercentage",                 0.01, 0.00, 0.5 );
DeclareColourParam( FGColour, "Colour",  kNoGroup, kNoFlags, 1.0, 1.0, 1.0, 1.0 );

DeclareFloatParam( _OutputAspectRatio );
DeclareFloat4Param( _bgExtents );

//--------------------------------------------------------------
// Inputs
//--------------------------------------------------------------
DeclareInput( bg );

//--------------------------------------------------------------
// Code
//--------------------------------------------------------------
DeclareEntryPoint( rectangle_main )
{
   float4 bgPix = ReadPixel( bg, uv1 );

   // Calc exact rectangle bounds

   float l = CentreX - ( Width / _OutputAspectRatio / 2.0 );
   float r = l + Width / _OutputAspectRatio;
   float t = ( 1.0 - CentreY ) - ( Height / 2.0 );
   float b = t + Height;

   float4 ret;

   // Remap the supplied pos so that it's in OUTPUT space, not INPUT space :

   float2 uv = float2(
      ( uv1.x - _bgExtents[ 0 ] ) / ( _bgExtents[ 2 ] - _bgExtents[ 0 ] ),
      ( uv1.y - _bgExtents[ 1 ] ) / ( _bgExtents[ 3 ] - _bgExtents[ 1 ] ) );

   if ( uv.x >= l && uv.x <= r && uv.y >= t && uv.y <= b )
   {
      ret = lerp( bgPix, FGColour, FGColour.a );
   }
   else
   {
      float2 softness = float2( Softness / _OutputAspectRatio, Softness );

      // calc outer bounds

      float l2 = l - softness.x;
      float r2 = r + softness.x;
      float t2 = t - softness.y;
      float b2 = b + softness.y;

      if ( uv.x < l2 || uv.x > r2 || uv.y < t2 || uv.y > b2 )
      {
         ret = bgPix;
      }
      else
      {
         float amt = 1.0;

         if ( uv.x < l )
         {
            if ( uv.y < t )
            {
               amt = 1.0 - ( length( ( uv - float2( l, t ) ) / softness ) );
            }
            else if ( uv.y > b )
            {
               amt = 1.0 - ( length( ( uv - float2( l, b ) ) / softness ) );
            }
            else
               amt = ( uv.x - l2 ) / softness.x;
         }
         else if ( uv.x > r )
         {
            if ( uv.y < t )
            {
               amt = 1.0 - ( length( ( uv - float2( r, t ) ) / softness ) );
            }
            else if ( uv.y > b )
            {
               amt = 1.0 - ( length( ( uv - float2( r, b ) ) / softness ) );
            }
            else
               amt = ( r2 - uv.x ) / softness.x;
         }
         else if ( uv.y < t )
         {
            amt = ( uv.y - t2 ) / softness.y;
         }
         else
         {
            amt = ( b2 - uv.y ) / softness.y;
         }

         ret = lerp( bgPix, FGColour, max( amt, 0.0 ) * FGColour.a );
      }
   }

   return ret;
}

//--------------------------------------------------------------//
DeclareEntryPoint( ellipse_main )
{
   float4 bgPix = ReadPixel( bg, uv1 );

   float a = ( Width/ _OutputAspectRatio ) / 2;
   float b = Height / 2;

   float sa = ( ( Width / _OutputAspectRatio ) / 2 ) + ( Softness / _OutputAspectRatio );
   float sb = ( Height / 2 ) + Softness;

   float2 pos = uv1 - float2( CentreX, 1.0 - CentreY );

   // https://www.mathwarehouse.com/ellipse/equation-of-ellipse.php
   //
   // ( ( x * x ) / ( a * a ) ) + ( ( y * y ) / ( b * b ) ) = 1
   //

   float2 posSq = pos * pos;

   float v1 = ( posSq.x / ( a * a ) )   + ( posSq.y / ( b * b ) );
   float v2 = ( posSq.x / ( sa * sa ) ) + ( posSq.y / ( sb * sb ) );

   float4 ret;

   if ( v1 < 1.0 )
   {
      ret = lerp( bgPix, FGColour, FGColour.a );
   }
   else if ( v2 > 1.0 )
   {
      ret = bgPix;
   }
   else
   {
      // http://www.slader.com/discussion/question/what-is-the-equation-of-an-ellipse-in-polar-coordinates/

      float theta = atan2( pos.y, pos.x );

      float cosThetaSq = pow( cos( theta ), 2 );
      float sinThetaSq = pow( sin( theta ), 2 );

      float dLower = ( a * b )   / sqrt( (  b *  b * cosThetaSq ) + (  a *  a * sinThetaSq ) );
      float dUpper = ( sa * sb ) / sqrt( ( sb * sb * cosThetaSq ) + ( sa * sa * sinThetaSq ) );
      float d      = length( pos );

      ret = lerp( bgPix, FGColour, ( 1.0 - ( ( d - dLower ) / ( dUpper - dLower ) ) ) * FGColour.a );
   }

   return ret;
}
