//--------------------------------------------------------------//
// HSL Colour Correction
//
// Copyright (c) LWKS Software Ltd.  All Rights Reserved
//--------------------------------------------------------------//

int _LwksEffectInfo
<
   string EffectGroup = "GenericPixelShader";
   string Description = "HSV Colour Correction";
   string Category    = "Colour Effects";
   bool CanSize       = true;
> = 0;

//--------------------------------------------------------------//
// Params
//--------------------------------------------------------------//
float HueScale
<
   string Description = "Scale";
   string Group = "Hue";
   float MinVal = 0.00;
   float MaxVal = 5.00;
> = 1.00;

float HueBias
<
   string Description = "Bias";
   string Group = "Hue";
   float MinVal = 0.00;
   float MaxVal = 1.00;
> = 0.00;

float SaturationScale
<
   string Description = "Scale";
   string Group = "Saturation";
   float MinVal = 0.00;
   float MaxVal = 10.00;
> = 1.0;

float SaturationBias
<
   string Description = "Bias";
   string Group = "Saturation";
   float MinVal = -1.00;
   float MaxVal = 1.00;
> = 0.0;

float IntensityScale
<
   string Description = "Scale";
   string Group = "Value";
   float MinVal = 0.00;
   float MaxVal = 5.00;
> = 1.0;

float IntensityBias
<
   string Description = "Bias";
   string Group = "Value";
   float MinVal = -2.00;
   float MaxVal = 2.00;
> = 0.0;

texture Input;

sampler InputSampler = sampler_state
{
   Texture  = <Input>;
   AddressU  = Border;
   AddressV  = Border;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;
};

texture HSLTex : RenderColorTarget < float2 ViewportRatio={1.0,1.0}; >;
sampler HSLSampler = sampler_state { Texture = <HSLTex>; };

// PI * 2
const float _PI2 = 6.2831853;

//--------------------------------------------------------------//
// HSLtoRGB
//--------------------------------------------------------------//
float4 RGBtoHSL_ps_main( float2 texCoord : TEXCOORD1 ) : COLOR
{
   float4 rgba = tex2D( InputSampler, texCoord );

   float r = rgba.r;
   float g = rgba.g;
   float b = rgba.b;

   float rg = r - g;
   float rb = r - b;

   // Hue
   float h = acos((rg + rb) / (2 * sqrt(rg * rg + rb * (g - b)))) / _PI2;

   if (b > g)
      h = 1 - h;

   // Intensity
   float i = (r + g + b) / 3;

   // Saturation
   float s = 1 - min(min(r, g), b) / i;

   return float4( h,s,i, rgba.a );
}

//--------------------------------------------------------------//
// HSLtoRGB
//--------------------------------------------------------------//
const float _rt3 = 1.0 / sqrt( 3.0 );

float4 HSLtoRGB_ps_main( float2 xy1 : TEXCOORD1 ) : COLOR
{
   float4 hsi = tex2D( HSLSampler, xy1 );
   float4 src = tex2D( InputSampler, xy1 );

   // Scale and bias our components
   float h = frac( hsi.r * HueScale + HueBias);
   float s = saturate(hsi.g * SaturationScale + SaturationBias);
   float i = saturate(hsi.b * IntensityScale  + IntensityBias);

   float h3 = 3 * h;
   float x = (2 * floor(h3) + 1) / 6;

   // Get our rgb components
   float r = (1 - s) * i;
   float H = _rt3 * tan((h - x) * _PI2);
   float b = ((3 + 3 * H) * i - (1 + 3 * H) * r) / 2;
   float g = 3 * i - b - r;

   // Put them in right order
   float3 rgb;
   if (h3 < 1)
   {
      rgb = float3(g, b, r);
   }
   else if (h3 < 2)
   {
      rgb = float3(r, g, b);
   }
   else
   {
      rgb = float3(b, r, g);
   }

   float4 correctedPixel = float4( rgb, 1.0 );

   float4 ret = lerp( src, correctedPixel, hsi.a );
   ret.a = hsi.a;

   return ret;
}

//--------------------------------------------------------------//
// techniques
//--------------------------------------------------------------//
technique HSL
{
   pass RGBtoHSL
   <
      string Script = "RenderColorTarget0 = HSLTex;";
   >
   {
      PixelShader = compile PROFILE RGBtoHSL_ps_main();
   }

   pass HSLtoRGB
   {
      PixelShader = compile PROFILE HSLtoRGB_ps_main();
   }
}
