//--------------------------------------------------------------
// ChromaKey3
//
// Copyright (c) LWKS Software Ltd.  All Rights Reserved
//--------------------------------------------------------------
#include "_utils.fx"

DeclareLightworksEffect( "Chromakey", "Keying", "Components", kNoNotes, "CanSize" );

//--------------------------------------------------------------
// Params
//--------------------------------------------------------------
DeclareColourParam( KeyColour, "Key Colour", kNoGroup, "SpecifiesColourRange", 235, 0.8, 0.8, -1 );
DeclareColourParam( Tolerance, "Tolerance",  kNoGroup, "SpecifiesColourRange|Hidden", 11, 0.2, 0.2, -1 );
DeclareColourParam( ToleranceSoftness, "Tolerance softness", kNoGroup, "SpecifiesColourRange|Hidden", 10, 0.1, 0.1, -1 );
DeclareFloatParam( KeySoftAmount, "Key softness", kNoGroup, kNoFlags, 0.5, 0.0, 1.0 );
DeclareFloatParam( RemoveSpill, "Remove spill", kNoGroup, kNoFlags, 0.5, 0.0, 1.0 );
DeclareBoolParam( Invert, "Invert", kNoGroup, false );
DeclareBoolParam( Reveal, "Reveal", kNoGroup, false );

//--------------------------------------------------------------
// Inputs
//--------------------------------------------------------------
DeclareInputs( fg, bg );
DeclareMask;

//--------------------------------------------------------------
// Common
//--------------------------------------------------------------
float _FallOff = 0.12;

DeclareFloatParam( _OutputWidth );
DeclareFloatParam( _OutputHeight );

#define HUE_IDX 0
#define SAT_IDX 1
#define VAL_IDX 2

float _oneSixth = 1.0 / 6.0;
float _minTolerance = 1.0 / 256.0;
float blur[] = { 20.0 / 64.0, 15.0 / 64.0, 6.0  / 64.0, 1.0  / 64.0 };  // See Pascals Triangle

bool allPositive( float4 pixel )
{
   return pixel.r > 0 && pixel.g > 0 && pixel.b > 0;
}

//--------------------------------------------------------------
// keygen
//
// Convert the source to HSV and then compute its similarity with
// the specified key-colour
//--------------------------------------------------------------//
DeclarePass( keygen )
{
   float keyVal = 1.0;
   float hueSimilarity = 1.0;

   float4 tolerance1 = Tolerance + _minTolerance;
   float4 tolerance2 = tolerance1 + ToleranceSoftness;

   float4 hsva = float4( 0.0, 0.0, 0.0, 0.0 );
   float4 rgba = ReadPixel( fg, uv1 );

   float maxComponentVal = max( max( rgba.r, rgba.g ), rgba.b );
   float minComponentVal = min( min( rgba.r, rgba.g ), rgba.b );
   float componentRange  = maxComponentVal - minComponentVal;

   hsva[ VAL_IDX ] = maxComponentVal;
   hsva[ SAT_IDX ] = componentRange / maxComponentVal;

   if ( hsva[ SAT_IDX ] == 0 )
   {
      hsva[ HUE_IDX ] = 0;   // undefined
   }
   else
   {
      if ( rgba.r == maxComponentVal )
      {
         hsva[ HUE_IDX ] = ( rgba.g - rgba.b ) / componentRange;
      }
      else if ( rgba.g == maxComponentVal )
      {
         hsva[ HUE_IDX ] = 2 + ( ( rgba.b - rgba.r ) / componentRange );
      }
      else
      {
         hsva[ HUE_IDX ] = 4 + ( ( rgba.r - rgba.g ) / componentRange );
      }

      hsva[ HUE_IDX ] *= _oneSixth;
      if (hsva[ HUE_IDX ] < 0) hsva[ HUE_IDX ] += 1.0;
   }

   // Calc difference between current pixel and specified key-colour
   float4 diff = abs( hsva - KeyColour );

   if ( diff[ HUE_IDX ] > 0.5 )
      diff[ HUE_IDX ] = 1.0 - diff[ HUE_IDX ];

   // Work out how transparent/opaque the corrected pixel will be
   if ( allPositive( tolerance2 - diff ) )
   {
      if ( allPositive( tolerance1 - diff ) )
      {
         keyVal = 0.0;
      }
      else
      {
         diff -= tolerance1;
         hueSimilarity = diff[ HUE_IDX ];
         diff /= ToleranceSoftness;
         keyVal = max( diff[ HUE_IDX ], max( diff[ SAT_IDX ], diff[ VAL_IDX ] ) );
         keyVal = pow( keyVal, 0.25 );
      }
   }
   else
   {
      diff -= tolerance1;
      hueSimilarity = diff[ HUE_IDX ];
   }

   return float4( keyVal, keyVal, keyVal, 1.0 - hueSimilarity );
}

//--------------------------------------------------------------
// Blur1
//--------------------------------------------------------------
DeclarePass( blur1 )
{
   float2 onePixAcross   = float2( KeySoftAmount / _OutputWidth, 0.0 );
   float2 twoPixAcross   = onePixAcross * 2.0;
   float2 threePixAcross = onePixAcross * 3.0;

   float4 result = tex2D( keygen, uv3 );

   result.r *= blur[ 0 ];
   result.r += tex2D( keygen, uv3 + onePixAcross ).r   * blur[ 1 ];
   result.r += tex2D( keygen, uv3 - onePixAcross ).r   * blur[ 1 ];
   result.r += tex2D( keygen, uv3 + twoPixAcross ).r   * blur[ 2 ];
   result.r += tex2D( keygen, uv3 - twoPixAcross ).r   * blur[ 2 ];
   result.r += tex2D( keygen, uv3 + threePixAcross ).r * blur[ 3 ];
   result.r += tex2D( keygen, uv3 - threePixAcross ).r * blur[ 3 ];

   return result;
}


//--------------------------------------------------------------
// Blur2
//--------------------------------------------------------------
DeclarePass( blur2 )
{
   float2 onePixDown   = float2( 0.0, KeySoftAmount / _OutputHeight );
   float2 twoPixDown   = onePixDown * 2.0;
   float2 threePixDown = onePixDown * 3.0;

   float4 result = tex2D( blur1, uv3 );

   result.r *= blur[ 0 ];
   result.r += tex2D( blur1, uv3 + onePixDown ).r   * blur[ 1 ];
   result.r += tex2D( blur1, uv3 - onePixDown ).r   * blur[ 1 ];
   result.r += tex2D( blur1, uv3 + twoPixDown ).r   * blur[ 2 ];
   result.r += tex2D( blur1, uv3 - twoPixDown ).r   * blur[ 2 ];
   result.r += tex2D( blur1, uv3 + threePixDown ).r * blur[ 3 ];
   result.r += tex2D( blur1, uv3 - threePixDown ).r * blur[ 3 ];

   return result;
}

//--------------------------------------------------------------//
// Composite
//
// Blend the foreground with the background using the key that
// was built in keygen_ps_main.  Apply spill-suppression as we go.
//--------------------------------------------------------------//
DeclareEntryPoint( composite )
{
   float4 result;

   float4 fgPix = ReadPixel( fg, uv1 );
   float4 key   = tex2D( blur2, uv3 );

//   float4 bgPix = ReadPixel( bg, uv2 );
   float4 bgPix = lerp( tex2D( bg, uv2 ), float4( 0,0,0,1 ), any( saturate( uv2 ) - uv2 ) ); // ReadPixelOpaque

   // key.r = blurred key
   // key.g = raw, unblurred key
   // key.a = spill removal amount

   // Using min( key.r, key.g ) means that any softness around
   // the key causes the foreground to shrink in from the edges
   float mixAmount = saturate( ( 1.0 - min( key.r, key.g ) * fgPix.a ) * 2 );

   if ( Reveal )
   {
      mixAmount = lerp( mixAmount, 1.0 - mixAmount, Invert );
      result = float4( mixAmount, mixAmount, mixAmount, 1.0 );
   }
   else if ( Invert )
   {
      result = lerp( bgPix, fgPix, mixAmount * bgPix.a );
      result.a = max( bgPix.a, mixAmount );
   }
   else
   {
      if ( key.a > 0.8 )
      {
         float lum = ( fgPix.r + fgPix.g + fgPix.b ) / 3.0;
         float4 fgPixLum = float4( lum, lum, lum, 1.0 );

         // Remove spill..
         fgPix = lerp( fgPix, fgPixLum, ( ( key.a - 0.8 ) / 0.2 ) * RemoveSpill );
      }

      fgPix.a = 1.0 - mixAmount;
      result = float4( lerp( bgPix, fgPix, fgPix.a ).rgb, max( bgPix.a, fgPix.a ) );
   }

   return lerp( bgPix, result, tex2D( Mask, uv1 ) );
}
