//--------------------------------------------------------------//
// Colour Correction 4
//
// Copyright (c) LWKS Software Ltd.  All Rights Reserved
//--------------------------------------------------------------//
DeclareLightworksEffect( "Colour Correction", "Colour", "Components", kNoNotes, CanSize );

//--------------------------------------------------------------
// Params
//--------------------------------------------------------------
DeclareColourParam( BlackLevel,       "Black Level", kNoGroup, kNoFlags, 0,0,0,1 );
DeclareColourParam( GreyLevel,        "Grey Level",  kNoGroup, kNoFlags, 0.5,0.5,0.5,1 );
DeclareColourParam( WhiteLevel ,      "White Level", kNoGroup, kNoFlags, 1,1,1,1 );
DeclareColourParam( ShadowTintColour, "Black Level", kNoGroup, kNoFlags, 0,0,0,1 );
DeclareColourParam( MidTintColour,    "Black Level", kNoGroup, kNoFlags, 0,0,0,1 );
DeclareColourParam( HighTintColour,   "Black Level", kNoGroup, kNoFlags, 0,0,0,1 );
DeclareFloatParam( Saturation,        "Saturation",  kNoGroup, kNoFlags, 0,0,0 );
DeclareColourParam( Gamma,            "Gamma",       kNoGroup, kNoFlags, 0,0,0,0 );
DeclareColourParam( Contrast,         "Contrast",    kNoGroup, kNoFlags, 0,0,0,0 );
DeclareColourParam( Brightness,       "Brightness",  kNoGroup, kNoFlags, 0,0,0,0 );
DeclareColourParam( Gain,             "Gain",        kNoGroup, kNoFlags, 0,0,0,0 );
DeclareFloatParam( HueBias,           "Hue Bias",    kNoGroup, kNoFlags, 0,0,0 );
DeclareFloatParam( SatBias,           "Sat Bias",    kNoGroup, kNoFlags, 0,0,0 );
DeclareFloatParam( ValBias,           "Val Bias",    kNoGroup, kNoFlags, 0,0,0 );
DeclareFloatParam( SatScale,          "Sat scale",   kNoGroup, kNoFlags, 0,0,0 );
DeclareFloatParam( ValScale,          "Val scale ",  kNoGroup, kNoFlags, 0,0,0 );

float _pi     = 3.14159;
float _pi2    = 6.2831853;
float _offset = 0.5 / 256.0;
float _scale  = 255.0 / 256.0;
float _rt3    = 0.57735; // 1.0 / sqrt( 3.0 );

//--------------------------------------------------------------//
// Inputs
//--------------------------------------------------------------//
DeclareInput( Input );
DeclareMask;
DeclareInput( Lut );

//--------------------------------------------------------------
// RGBtoHSL
//--------------------------------------------------------------
float4 RGBtoHSL( float4 src )
{
   float4 ret = src;

   // Do balancing
   float greyAverage = ( GreyLevel.r + GreyLevel.g + GreyLevel.b ) / 3.0;
   ret -= BlackLevel;
   ret /= ( WhiteLevel - BlackLevel );
   ret -= GreyLevel;
   ret += greyAverage;
   ret = clamp( ret, 0.0, 1.0 );

   // Calc its brightness (so that it can be treated as a shadow/midtone/highlight as appropriate)
   float lum = ( ret.r + ret.g + ret.b ) / 3.0;

   // Apply the shadow/midtone/highlight tints
   float2 lutPos   = float2( ( lum * _scale ) + _offset, 0.5 );
   float4 lutPixel = tex2D( Lut, lutPos );
   ret += ShadowTintColour * lutPixel.r;
   ret += MidTintColour    * lutPixel.g;
   ret += HighTintColour   * lutPixel.b;

   float avg = ( ret.r + ret.g + ret.b ) / 3.0;
   float4 avg4 = float4( avg, avg, avg, avg );
   ret = avg4 + ( ( ret - avg4 ) * Saturation );

   // Do brightness, gamma & contrast logic
   ret = ( ( ( ( pow( saturate( ret ), 1 / Gamma ) * Gain ) + Brightness ) - 0.5 ) * Contrast ) + 0.5;

   float r  = saturate( ret.r );
   float g  = saturate( ret.g );
   float b  = saturate( ret.b );
   float rg = r - g;
   float rb = r - b;

   lum = ( r + g + b ) / 3.0;

   // Hue
   float h = acos( ( rg + rb ) / ( 2 * sqrt( rg * rg + rb * ( g - b ) ) ) ) / _pi2;

   if ( b > g )
      h = 1.0 - h;

   // Saturation
   float s = 1.0 - min( min( r, g ), b ) / lum;

   return float4( h, s, lum, src.a );
}

//--------------------------------------------------------------//
// HSLtoRGB
//--------------------------------------------------------------//
float4 HSLtoRGB( float4 hsi )
{
   float4 ret;

   // Scale and bias our components
   float h = frac( hsi.r + HueBias );
   float s = saturate( hsi.g * SatScale + SatBias );
   float i = saturate( hsi.b * ValScale  + ValBias );

   float h3 = 3 * h;
   float x = ( 2 * floor( h3 ) + 1 ) / 6;

   // Get our rgb components
   float r = ( 1 - s ) * i;
   float H = _rt3 * tan( ( h - x ) * _pi2 );
   float b = ( ( 3 + 3 * H ) * i - ( 1 + 3 * H ) * r ) / 2;
   float g = 3 * i - b - r;

   // Put them in right order
   if ( h3 < 1 )
   {
      ret = float4( g, b, r, hsi.a );
   }
   else if ( h3 < 2 )
   {
      ret = float4( r, g, b, hsi.a );
   }
   else
   {
      ret = float4( b, r, g, hsi.a );
   }

   return ret;
}

//--------------------------------------------------------------
// Entry points
//--------------------------------------------------------------
DeclareEntryPoint( CC )
{
   // Read the source image pixel
   float4 src = ReadPixel( Input, uv1 );
   float4 ret = src;

   // Do balancing
   float greyAverage = ( GreyLevel.r + GreyLevel.g + GreyLevel.b ) / 3.0;
   ret -= BlackLevel;
   ret /= ( WhiteLevel - BlackLevel );
   ret -= GreyLevel;
   ret += greyAverage;
   ret = clamp( ret, 0.0, 1.0 );

   // Calc its brightness (so that it can be treated as a shadow/midtone/highlight as appropriate)
   float lum = ( ret.r + ret.g + ret.b ) / 3.0;

   // Apply the shadow/midtone/highlight tints
   float2 lutPos   = float2( ( lum * _scale ) + _offset, 0.5 );
   float4 lutPixel = tex2D( Lut, lutPos );
   ret += ShadowTintColour * lutPixel.r;
   ret += MidTintColour    * lutPixel.g;
   ret += HighTintColour   * lutPixel.b;

   float avg = ( ret.r + ret.g + ret.b ) / 3.0;
   float4 avg4 = float4( avg, avg, avg, avg );
   ret = avg4 + ( ( ret - avg4 ) * Saturation );

   // Do brightness, gamma & contrast logic
   ret = ( ( ( ( pow( saturate( ret ), 1 / Gamma ) * Gain ) + Brightness ) - 0.5 ) * Contrast ) + 0.5;

   ret.a = src.a;

   return ret;
}
//--------------------------------------------------------------
DeclareEntryPoint( CCMasked )
{
   // Read the source image pixel
   float4 src = ReadPixel( Input, uv1 );
   float4 ret = src;

   // Do balancing
   float greyAverage = ( GreyLevel.r + GreyLevel.g + GreyLevel.b ) / 3.0;
   ret -= BlackLevel;
   ret /= ( WhiteLevel - BlackLevel );
   ret -= GreyLevel;
   ret += greyAverage;
   ret = clamp( ret, 0.0, 1.0 );

   // Calc its brightness (so that it can be treated as a shadow/midtone/highlight as appropriate)
   float lum = ( ret.r + ret.g + ret.b ) / 3.0;

   // Apply the shadow/midtone/highlight tints
   float2 lutPos   = float2( ( lum * _scale ) + _offset, 0.5 );
   float4 lutPixel = tex2D( Lut, lutPos );
   ret += ShadowTintColour * lutPixel.r;
   ret += MidTintColour    * lutPixel.g;
   ret += HighTintColour   * lutPixel.b;

   float avg = ( ret.r + ret.g + ret.b ) / 3.0;
   float4 avg4 = float4( avg, avg, avg, avg );
   ret = avg4 + ( ( ret - avg4 ) * Saturation );

   // Do brightness, gamma & contrast logic
   ret = ( ( ( ( pow( saturate( ret ), 1 / Gamma ) * Gain ) + Brightness ) - 0.5 ) * Contrast ) + 0.5;

   ret.a = src.a;

   return lerp( src, ret, tex2D( Mask, uv2 ) );
}
//--------------------------------------------------------------
DeclareEntryPoint( H2R2H )
{
   return HSLtoRGB( RGBtoHSL( ReadPixel( Input, uv1 ) ) );
}
//--------------------------------------------------------------
DeclareEntryPoint( H2R2HMasked )
{
   float4 src = ReadPixel( Input, uv1 );
   float4 ret = HSLtoRGB( RGBtoHSL( src ) );
   return lerp( src, ret, tex2D( Mask, uv2 ) );
}
