//--------------------------------------------------------------//
// Blur
//
// Shader used in combination with image-reduction to produce
// large blurs
//
// Copyright (c) LWKS Software Ltd.  All Rights Reserved
//--------------------------------------------------------------//
#include "_utils.fx"

int _LwksEffectInfo
<
   string EffectGroup = "GenericPixelShader";
   string Description = "Fast Blur";
   string Category    = "Stylize";
   string Notes       = "Fast but simple blur";
   bool CanSize       = false;
> = 0;

//--------------------------------------------------------------//
// Params
//--------------------------------------------------------------//

float Amount
<
   string Description = "Amount";
   float MinVal = 0.00;
   float MaxVal = 1.00;
> = 0.5;

float _OutputPixelWidth  = 1.0;
float _OutputPixelHeight = 1.0;
float _MaxXPos           = 1.0;
float _MaxYPos           = 1.0;
float _MinXPos           = 0.0;
float _MinYPos           = 0.0;

//--------------------------------------------------------------//
// Inputs
//--------------------------------------------------------------//
DeclareInput( Input, InputSampler );

texture Intermediate : RenderColorTarget;
sampler IntermediateSampler = sampler_state { Texture = <Intermediate>; };


//--------------------------------------------------------------//
float safeX( float x )
{
   return max( min( x, _MaxXPos ), _MinXPos );
}
//--------------------------------------------------------------//
float safeY( float y )
{
   return max( min( y, _MaxYPos ), _MinYPos );
}

//--------------------------------------------------------------//
// Code
//--------------------------------------------------------------//

float4 blurX_main( float2 xy1 : TEXCOORD1 ) : COLOR
{
   float y = safeY( xy1.y );
   float twoPixels = _OutputPixelWidth * 2;
   float halfPixel = _OutputPixelWidth / 2;

   float tap1  = safeX( xy1.x + halfPixel );
   float tap2  = safeX( tap1 + twoPixels );
   float tap3  = safeX( tap2 + twoPixels );
   float ntap1 = safeX( xy1.x - ( halfPixel + _OutputPixelWidth ) );
   float ntap2 = safeX( ntap1 - twoPixels );
   float ntap3 = safeX( ntap2 - twoPixels );

   float4 ret  = tex2D( InputSampler, float2( tap1,  y ) );

   float4 blurred = ret;
   blurred += tex2D( InputSampler, float2( tap2,  y ) );
   blurred += tex2D( InputSampler, float2( tap3,  y ) );
   blurred += tex2D( InputSampler, float2( ntap1, y ) );
   blurred += tex2D( InputSampler, float2( ntap2, y ) );
   blurred += tex2D( InputSampler, float2( ntap3, y ) );
   blurred /= 6;

   ret.rgb = blurred.rgb;

   return ret;
}

//--------------------------------------------------------------//
float4 blurY_main( float2 xy1 : TEXCOORD1 ) : COLOR
{
   float x = safeX( xy1.x );
   float twoPixels = _OutputPixelHeight * 2;
   float halfPixel = _OutputPixelHeight / 2;

   float tap1  = safeY( xy1.y + halfPixel );
   float tap2  = safeY( tap1 + twoPixels );
   float tap3  = safeY( tap2 + twoPixels );
   float ntap1 = safeY( xy1.y - ( halfPixel + _OutputPixelHeight ) );
   float ntap2 = safeY( ntap1 - twoPixels );
   float ntap3 = safeY( ntap2 - twoPixels );

   float4 ret  = tex2D( IntermediateSampler, float2( x, tap1 ) );
   float4 orig = tex2D( InputSampler, xy1 );

   float4 blurred = ret;
   blurred += tex2D( IntermediateSampler, float2( x, tap2  ) );
   blurred += tex2D( IntermediateSampler, float2( x, tap3  ) );
   blurred += tex2D( IntermediateSampler, float2( x, ntap1 ) );
   blurred += tex2D( IntermediateSampler, float2( x, ntap2 ) );
   blurred += tex2D( IntermediateSampler, float2( x, ntap3 ) );
   blurred /= 6;

   ret.rgb = blurred.rgb;

   return ret;
}

//--------------------------------------------------------------
// Alpha combine
//--------------------------------------------------------------
texture _Original;
texture _Blurred;
DeclareInput( _mask, MaskSampler );

sampler BlurredSampler = sampler_state
{
   Texture = <_Blurred>;
   MINFILTER = LINEAR;
   MIPFILTER = LINEAR;
   MAGFILTER = LINEAR;
   ADDRESSU  = ClampToEdge;
   ADDRESSV  = ClampToEdge;
};

sampler OriginalSampler = sampler_state
{
   Texture = <_Original>;
   MINFILTER = LINEAR;
   MIPFILTER = LINEAR;
   MAGFILTER = LINEAR;
};

//--------------------------------------------------------------//
float4 alphaCombine_main( float2 xy1 : TEXCOORD1, float2 xy2 : TEXCOORD2 ) : COLOR
{
   float4 orig    = tex2D( OriginalSampler, xy1 );
   float4 blurred = tex2D( BlurredSampler,  xy2 );

   float4 ret = lerp( orig, blurred, orig.a );
   ret.a = orig.a;

   return lerp( orig, ret, tex2D(MaskSampler, xy1) );
}

//--------------------------------------------------------------//
technique Blur
{
   pass BlurX < string Script = "RenderColorTarget0 = Intermediate;"; >
   {
      PixelShader = compile PROFILE blurX_main();
   }

   pass BlurY
   {
      PixelShader = compile PROFILE blurY_main();
   }
}

//--------------------------------------------------------------//
technique AlphaCombine
{
   pass P1
   {
      PixelShader = compile PROFILE alphaCombine_main();
   }
}
