//-----------------------------------------------------------------------------------------//
// BigBlur2.fx
//
// Credits : khaver & schrauber
//-----------------------------------------------------------------------------------------//

#include "_utils.fx"

DeclareLightworksEffect( "Blur", "Stylize", "Components", "High quality blur using a rotating Poisson distribution", kNoFlags);

//-----------------------------------------------------------------------------------------//
// Inputs & Samplers
//-----------------------------------------------------------------------------------------//

DeclareInput (Input, Linear);

DeclareMask;

//-----------------------------------------------------------------------------------------//
// Parameters
//-----------------------------------------------------------------------------------------//

DeclareFloatParam (Size, "Radius", kNoGroup, kNoFlags, 0.1, 0.0, 1.0);

DeclareIntParam (SetTechnique, "Blur strength", kNoGroup, 0, "Normal|Fine");

//-----------------------------------------------------------------------------------------//
// Definitions and declarations
//-----------------------------------------------------------------------------------------//

DeclareFloatParam (_OutputAspectRatio);

#define PI_HALF      1.5708  // 90 degrees in radians
#define PI_SIXTH     0.5236  // 30 degrees in radians

#define OFFSET_1     0.0873  // 5 degrees in radians
#define OFFSET_2     0.1745  // 10
#define OFFSET_3     0.2618  // 15
#define OFFSET_4     0.3491  // 20
#define OFFSET_5     0.4363  // 25

#define OFFSET_1_2   0.1309  // 7.5 degrees in radians
#define OFFSET_4_5   0.3927  // 22.5

#define NORMAL_SCALE 0.05
#define FINE_SCALE   0.006

//-----------------------------------------------------------------------------------------//
// Functions
//-----------------------------------------------------------------------------------------//

float4 bigBlur (float2 uv, sampler2D blurSampler, float offset, float scale)
{
   float4 cOut = tex2D (blurSampler, uv);

   float2 radius = float2 (1.0, _OutputAspectRatio) * Size * scale;
   float2 angle  = float2 (offset + PI_HALF, offset);
   float2 coord, sample;

   for (int tap = 0; tap < 12; tap++) {
      sample = sin (angle);
      coord  = uv + (sample * radius);
      cOut  += tex2D (blurSampler, coord);
      angle += PI_SIXTH;
   }

   return cOut / 13.0;
}

//-----------------------------------------------------------------------------------------//
// Shaders
//-----------------------------------------------------------------------------------------//

// Normal blur strength

DeclarePass (A_0)
{ return ReadPixel (Input, uv1); }

DeclarePass (A_1)
{
   return bigBlur (uv2, A_0, 0.0, NORMAL_SCALE);
}

DeclarePass (A_2)
{
   return bigBlur (uv2, A_1, OFFSET_1, NORMAL_SCALE);
}

DeclarePass (A_3)
{
   return bigBlur (uv2, A_2, OFFSET_2, NORMAL_SCALE);
}

DeclarePass (A_4)
{
   return bigBlur (uv2, A_3, OFFSET_3, NORMAL_SCALE);
}

DeclarePass (A_5)
{
   return bigBlur (uv2, A_4, OFFSET_4, NORMAL_SCALE);
}

DeclareEntryPoint (BigBlur)
{
   float4 Blur = bigBlur (uv2, A_5, OFFSET_5, NORMAL_SCALE);

   return lerp (ReadPixel (A_0, uv2), Blur, tex2D (Mask, uv1));
}

//-----------------------------------------------------------------------------------------//

// Fine blur strength

DeclarePass (B_0)
{ return ReadPixel (Input, uv1); }

DeclarePass (B_1)
{
   return bigBlur (uv2, B_0, 0.0, FINE_SCALE);
}

DeclarePass (B_2)
{
   return bigBlur (uv2, B_1, OFFSET_1_2, FINE_SCALE);
}

DeclarePass (B_3)
{
   return bigBlur (uv2, B_2, OFFSET_3, FINE_SCALE);
}

DeclareEntryPoint (FineBlur)
{
   float4 Blur = bigBlur (uv2, B_3, OFFSET_4_5, FINE_SCALE);

   return lerp (ReadPixel (B_0, uv2), Blur, tex2D (Mask, uv1));
}

